<?php
/**
 * Remote Upload API Endpoint
 * This file should be placed on the remote server
 * Example: https://test.glint.one/api/upload-image.php
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST');
header('Access-Control-Allow-Headers: Content-Type');

// Configuration for remote server
$config = [
    'api_key' => 'G21K07X91KRAR68JWE4FI8RXQBYZWHZH',
    'upload_path' => 'uploads/',
    'thumbnail_path' => 'uploads/thumbs/',
    'base_url' => 'https://test.glint.one/my/',
    'max_file_size' => 5 * 1024 * 1024, // 5MB
    'allowed_types' => ['jpg', 'jpeg', 'png', 'gif', 'webp']
];

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Check API key
$api_key = $_POST['api_key'] ?? '';
if ($api_key !== $config['api_key']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'Invalid API key']);
    exit;
}

// Handle different actions
$action = $_POST['action'] ?? 'upload';

if ($action === 'delete') {
    handleDelete($config);
    exit;
}

// Check if file was uploaded (only for upload action)
if ($action === 'upload' && (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK)) {
    echo json_encode(['success' => false, 'message' => 'No file uploaded']);
    exit;
}

$file = $_FILES['file'];
$width = intval($_POST['width'] ?? 0);
$height = intval($_POST['height'] ?? 0);

// Validate file
$validation = validateFile($file, $config);
if (!$validation['valid']) {
    echo json_encode(['success' => false, 'message' => $validation['message']]);
    exit;
}

// Process upload
$result = processUpload($file, $width, $height, $config);

if ($result['success']) {
    echo json_encode([
        'success' => true,
        'message' => 'File uploaded successfully',
        'image_url' => $result['image_url'],
        'thumbnail_url' => $result['thumbnail_url']
    ]);
} else {
    echo json_encode([
        'success' => false,
        'message' => $result['message']
    ]);
}

function validateFile($file, $config) {
    // Check file size
    if ($file['size'] > $config['max_file_size']) {
        return ['valid' => false, 'message' => 'File too large'];
    }
    
    // Check file type
    $fileType = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    if (!in_array($fileType, $config['allowed_types'])) {
        return ['valid' => false, 'message' => 'Invalid file type'];
    }
    
    // Check MIME type
    $finfo = finfo_open(FILEINFO_MIME_TYPE);
    $mimeType = finfo_file($finfo, $file['tmp_name']);
    finfo_close($finfo);
    
    $allowedMimes = [
        'image/jpeg',
        'image/png',
        'image/gif',
        'image/webp'
    ];
    
    if (!in_array($mimeType, $allowedMimes)) {
        return ['valid' => false, 'message' => 'Invalid file format'];
    }
    
    return ['valid' => true];
}

function processUpload($file, $width, $height, $config) {
    try {
        // Create upload directories
        if (!is_dir($config['upload_path'])) {
            mkdir($config['upload_path'], 0755, true);
        }
        if (!is_dir($config['thumbnail_path'])) {
            mkdir($config['thumbnail_path'], 0755, true);
        }
        
        // Generate unique filename
        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $uniqueId = uniqid() . '_' . time();
        $fileName = "uploaded_{$uniqueId}.{$fileExtension}";
        $thumbnailName = "thumb_{$uniqueId}.jpg";
        
        $uploadPath = $config['upload_path'] . $fileName;
        $thumbnailPath = $config['thumbnail_path'] . $thumbnailName;
        
        // Move uploaded file
        if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
            return ['success' => false, 'message' => 'Failed to save file'];
        }
        
        // Create thumbnail
        createThumbnail($uploadPath, $thumbnailPath, 300, 300);
        
        return [
            'success' => true,
            'image_url' => $config['base_url'] . $uploadPath,
            'thumbnail_url' => $config['base_url'] . $thumbnailPath
        ];
        
    } catch (Exception $e) {
        return [
            'success' => false,
            'message' => 'Upload failed: ' . $e->getMessage()
        ];
    }
}

function createThumbnail($sourcePath, $thumbnailPath, $maxWidth, $maxHeight) {
    $imageInfo = getimagesize($sourcePath);
    $width = $imageInfo[0];
    $height = $imageInfo[1];
    $mimeType = $imageInfo['mime'];
    
    // Create image resource
    switch ($mimeType) {
        case 'image/jpeg':
            $source = imagecreatefromjpeg($sourcePath);
            break;
        case 'image/png':
            $source = imagecreatefrompng($sourcePath);
            break;
        case 'image/gif':
            $source = imagecreatefromgif($sourcePath);
            break;
        case 'image/webp':
            $source = imagecreatefromwebp($sourcePath);
            break;
        default:
            return false;
    }
    
    if (!$source) {
        return false;
    }
    
    // Calculate thumbnail dimensions
    $ratio = min($maxWidth / $width, $maxHeight / $height);
    $newWidth = round($width * $ratio);
    $newHeight = round($height * $ratio);
    
    // Create thumbnail
    $thumbnail = imagecreatetruecolor($newWidth, $newHeight);
    
    // Preserve transparency
    if ($mimeType === 'image/png' || $mimeType === 'image/gif') {
        imagealphablending($thumbnail, false);
        imagesavealpha($thumbnail, true);
        $transparent = imagecolorallocatealpha($thumbnail, 255, 255, 255, 127);
        imagefilledrectangle($thumbnail, 0, 0, $newWidth, $newHeight, $transparent);
    }
    
    // Resize
    imagecopyresampled(
        $thumbnail, $source,
        0, 0, 0, 0,
        $newWidth, $newHeight,
        $width, $height
    );
    
    // Save as JPEG
    $success = imagejpeg($thumbnail, $thumbnailPath, 90);
    
    // Clean up
    imagedestroy($source);
    imagedestroy($thumbnail);
    
    return $success;
}

function handleDelete($config) {
    $image_url = $_POST['image_url'] ?? '';
    $thumbnail_url = $_POST['thumbnail_url'] ?? '';
    $file_name = $_POST['file_name'] ?? '';
    
    if (empty($image_url) && empty($thumbnail_url)) {
        echo json_encode(['success' => false, 'message' => 'No image URLs provided']);
        return;
    }
    
    $deleted_files = [];
    $errors = [];
    
    // Delete main image
    if (!empty($image_url)) {
        $image_path = getFilePathFromUrl($image_url, $config['base_url']);
        if ($image_path && file_exists($image_path)) {
            if (unlink($image_path)) {
                $deleted_files[] = 'main image';
            } else {
                $errors[] = 'Failed to delete main image';
            }
        }
    }
    
    // Delete thumbnail
    if (!empty($thumbnail_url)) {
        $thumbnail_path = getFilePathFromUrl($thumbnail_url, $config['base_url']);
        if ($thumbnail_path && file_exists($thumbnail_path)) {
            if (unlink($thumbnail_path)) {
                $deleted_files[] = 'thumbnail';
            } else {
                $errors[] = 'Failed to delete thumbnail';
            }
        }
    }
    
    if (empty($errors)) {
        echo json_encode([
            'success' => true,
            'message' => 'Files deleted successfully: ' . implode(', ', $deleted_files)
        ]);
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Some files could not be deleted: ' . implode(', ', $errors)
        ]);
    }
}

function getFilePathFromUrl($url, $base_url) {
    if (strpos($url, $base_url) === 0) {
        $relative_path = substr($url, strlen($base_url));
        return $relative_path;
    }
    return null;
}
?>
